import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Music2, Shield, Clock, Users, Award, Star, User } from 'lucide-react'
import { motion } from 'framer-motion'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedStyle, setSelectedStyle] = useState('')
  const [startDate, setStartDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const danceStyles = [
    { 
      id: 'hiphop', 
      name: 'Хип-хоп', 
      icon: '🎵',
      color: '#2C3E50',
      description: 'Энергичный уличный танец',
      duration: '12 занятий',
      level: 'Любой уровень'
    },
    { 
      id: 'bachata', 
      name: 'Бачата', 
      icon: '💃',
      color: '#34495E',
      description: 'Страстный латиноамериканский танец',
      duration: '10 занятий',
      level: 'Начинающий'
    },
    { 
      id: 'zumba', 
      name: 'Зумба', 
      icon: '✨',
      color: '#2C3E50',
      description: 'Фитнес-танец для всех',
      duration: '8 занятий',
      level: 'Любой уровень'
    },
    { 
      id: 'ballroom', 
      name: 'Бальные', 
      icon: '🎩',
      color: '#34495E',
      description: 'Классические бальные танцы',
      duration: '16 занятий',
      level: 'Продвинутый'
    }
  ]

  const instructors = [
    { name: 'Алексей Воронов', experience: '10 лет', rating: 5, specialty: 'Хип-хоп, Брейк-данс', students: '3000+' },
    { name: 'Мария Соколова', experience: '12 лет', rating: 5, specialty: 'Бачата, Сальса', students: '4500+' },
    { name: 'Елена Петрова', experience: '8 лет', rating: 5, specialty: 'Зумба, Фитнес', students: '6000+' },
    { name: 'Дмитрий Кузнецов', experience: '15 лет', rating: 5, specialty: 'Бальные, Латина', students: '2500+' }
  ]

  const guarantees = [
    { icon: Shield, title: 'Гарантия качества', text: 'Профессиональные преподаватели с опытом' },
    { icon: Clock, title: 'Гибкий график', text: 'Занятия в удобное для вас время' },
    { icon: Users, title: 'Поддержка', text: 'Помощь на всех этапах обучения' },
    { icon: Award, title: 'Сертификат', text: 'Официальный документ по окончании' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedStyle && startDate) {
      const savedClasses = localStorage.getItem('danceClasses')
      const classes = savedClasses ? JSON.parse(savedClasses) : []
      
      const styleData = danceStyles.find(s => s.id === selectedStyle)
      const randomInstructor = instructors[Math.floor(Math.random() * instructors.length)]
      
      const newClass = {
        id: Date.now(),
        style: styleData.name,
        instructor: randomInstructor.name,
        date: startDate,
        rating: 0
      }

      classes.push(newClass)
      localStorage.setItem('danceClasses', JSON.stringify(classes))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedStyle('')
        setStartDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="enroll-page">
      <div className="container">
        <motion.section 
          className="enroll-hero"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <h1 className="enroll-title">Записаться на занятие</h1>
          <p className="enroll-subtitle">
            Выберите стиль танца и дату начала
          </p>
        </motion.section>

        <motion.section 
          className="enrollment-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          <div className="enrollment-form-wrapper">
            <form className="enrollment-form" onSubmit={handleSubmit}>
              <div className="form-group">
                <label className="form-label">
                  Выберите стиль танца
                </label>
                <div className="styles-list">
                  {danceStyles.map((style) => (
                    <motion.button
                      key={style.id}
                      type="button"
                      className={`style-item ${selectedStyle === style.id ? 'selected' : ''}`}
                      onClick={() => setSelectedStyle(style.id)}
                      whileHover={{ x: 4, transition: { duration: 0.2 } }}
                      whileTap={{ scale: 0.98 }}
                    >
                      <div className="style-icon">{style.icon}</div>
                      <div className="style-content">
                        <h3 className="style-name">{style.name}</h3>
                        <p className="style-description">{style.description}</p>
                        <div className="style-meta">
                          <span className="style-duration">
                            <Clock size={16} strokeWidth={2} />
                            {style.duration}
                          </span>
                          <span className="style-level">
                            {style.level}
                          </span>
                        </div>
                      </div>
                      {selectedStyle === style.id && (
                        <div className="style-check">
                          <CheckCircle size={24} strokeWidth={2} />
                        </div>
                      )}
                    </motion.button>
                  ))}
                </div>
              </div>

              <div className="form-group">
                <label for="startDate" className="form-label">
                  Дата начала занятий
                </label>
                <input
                  type="date"
                  id="startDate"
                  className="form-input"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  min={minDate}
                  max={maxDate}
                  required
                />
              </div>

              <motion.button 
                type="submit" 
                className="submit-button" 
                disabled={!selectedStyle || !startDate}
                whileHover={{ y: -2 }}
                whileTap={{ scale: 0.98 }}
              >
                <span>Записаться на занятие</span>
                <CheckCircle size={20} strokeWidth={2} className="btn-icon" />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="success-message"
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.3 }}
                >
                  <CheckCircle size={24} strokeWidth={2} className="success-icon" />
                  <span>Вы успешно записались на занятие!</span>
                </motion.div>
              )}
            </form>
          </div>
        </motion.section>

        <motion.section 
          className="guarantees-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.4 }}
        >
          <h2 className="section-title">Гарантии</h2>
          <div className="guarantees-grid">
            {guarantees.map((guarantee, index) => {
              const IconComponent = guarantee.icon
              return (
                <motion.div
                  key={guarantee.title}
                  className="guarantee-item"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.4 }}
                  whileHover={{ y: -4, transition: { duration: 0.2 } }}
                >
                  <div className="guarantee-icon">
                    <IconComponent size={32} strokeWidth={2} />
                  </div>
                  <h3 className="guarantee-title">{guarantee.title}</h3>
                  <p className="guarantee-text">{guarantee.text}</p>
                </motion.div>
              )
            })}
          </div>
        </motion.section>

        <motion.section 
          className="instructors-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.5 }}
        >
          <h2 className="section-title">Наши преподаватели</h2>
          <div className="instructors-grid">
            {instructors.map((instructor, index) => (
              <motion.div
                key={instructor.name}
                className="instructor-card"
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1, duration: 0.4 }}
                whileHover={{ y: -4, transition: { duration: 0.2 } }}
              >
                <div className="instructor-avatar">
                  <User size={32} strokeWidth={2} />
                </div>
                <h3 className="instructor-name">{instructor.name}</h3>
                <p className="instructor-experience">{instructor.experience} опыта</p>
                <div className="instructor-rating">
                  {Array.from({ length: 5 }, (_, i) => (
                    <Star key={i} size={16} fill="#2C3E50" stroke="#2C3E50" strokeWidth={2} />
                  ))}
                </div>
                <p className="instructor-specialty">{instructor.specialty}</p>
                <div className="instructor-students">
                  <Users size={16} strokeWidth={2} />
                  <span>{instructor.students} учеников</span>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default Enroll

